/*
  Author: Heath Cissell
  Date created: 4/19/17
  Program: Project EASI microcontroller code
  This work has been compiled using many sources mainly posts/wiki posts.
  The specific functions that are based on code from these sources are commented crediting 
  the appropriate person.
*/

#include "Wire.h"
#include "Adafruit_DRV2605.h"
#include <Arduino.h>

#include "Adafruit_BLE.h"

#include "Adafruit_BluefruitLE_UART.h"

#define TCAADDR 0x70
Adafruit_DRV2605 drv;

#define FACTORYRESET_ENABLE         1
#define MINIMUM_FIRMWARE_VERSION    "0.6.6"
#define MODE_LED_BEHAVIOUR          "MODE"

#define BUFSIZE                        128   // Size of the read buffer for incoming data
#define VERBOSE_MODE                   true  // If set to 'true' enables debug output

// Serial2 pin and pad definitions (in Arduino files Variant.h & Variant.cpp)
#define PIN_SERIAL2_RX       (34ul)               // Pin description number for PIO_SERCOM on D12
#define PIN_SERIAL2_TX       (36ul)               // Pin description number for PIO_SERCOM on D10
#define PAD_SERIAL2_TX       (UART_TX_PAD_2)      // SERCOM pad 2
#define PAD_SERIAL2_RX       (SERCOM_RX_PAD_3)    // SERCOM pad 3

// Instantiate the Serial2 class
Uart Serial2(&sercom1, PIN_SERIAL2_RX, PIN_SERIAL2_TX, PAD_SERIAL2_RX, PAD_SERIAL2_TX);
void SERCOM1_Handler()    // Interrupt handler for SERCOM1
{
  Serial2.IrqHandler();
}

//Create the hardware serial bluetooth object.
Adafruit_BluefruitLE_UART ble(Serial2, A3);

void error(const __FlashStringHelper*err) 
{
  Serial.println(err);
  while (1);
  
}

//Setting pin assignments for the different components
//  used in the system.
const int distSensor = 4;
const int leftSensor = 5;
const int rightSensor = 6;

const int distPower = A0;
const int leftPower = A1;
const int rightPower = A2;

const int sole0 = 2;
const int sole1 = 3;
const int sole2 = 7;
const int sole3 = 9;
const int sole4 = 11;
const int sole5 = 8;
const int sole6 = 13;

const int arraysize = 5;  //quantity of values to find the median (sample size). Needs to be an odd number
int rangevalue[arraysize];  //declare an array to store the sensor samples.
long pulse;               //Used when getting new data from the sensors.

//These values are used to set the pin states for the pin of each solenoid.
bool solStates[7] = {0,0,0,0,0,0,0};
int solMap[7] = {sole0,sole1,sole2,sole3,sole4,sole5,sole6};

//These values are used to keep track of the previous distance that the sensors detected an object
//  so that repeated values will not be sent to the Bluetooth module.
int prevDist = -1;
int prevLeft = -1;
int prevRight = -1;

void setup()
{
  //Open up a serial connection
  Serial.begin(9600);

  //Wait for the serial connection
  delay(500);

  //Set mode of output pins
  pinMode(distPower, OUTPUT);
  pinMode(leftPower, OUTPUT);
  pinMode(rightPower, OUTPUT);
  pinMode(sole0, OUTPUT);     //Solenoid #1 - Distance: 6ft-9ft range
  pinMode(sole1, OUTPUT);     //Solenoid #2 - Distance: 3ft-6ft range
  pinMode(sole2, OUTPUT);     //Solenoid #3 - Distance: 0ft-3ft range
  pinMode(sole3, OUTPUT);     //Solenoid #4 - Direction: left
  pinMode(sole4, OUTPUT);     //Solenoid #5 - Direction: right
  pinMode(sole5, OUTPUT);     //Solenoid #6 - Navigation: left
  pinMode(sole6, OUTPUT);     //Solenoid #7 - Navigation: right

  //Setup Bluetooth module.
  if ( !ble.begin(VERBOSE_MODE) )
  {
    error(F("Couldn't find Bluefruit, make sure it's in CoMmanD mode & check wiring?"));
  }
  // Disable command echo from Bluefruit
  ble.echo(false);
  // Print Bluefruit information
  ble.info();
  ble.verbose(false);

  // LED Activity command is only supported from 0.6.6
  if ( ble.isVersionAtLeast(MINIMUM_FIRMWARE_VERSION) )
  {
    // Change Mode LED Activity
    ble.sendCommandCheckOK("AT+HWModeLED=" MODE_LED_BEHAVIOUR);
  }

  // Set module to DATA mode
  ble.setMode(BLUEFRUIT_MODE_DATA);

  //Setup I2c connections.
  Wire.begin();
  tcaselect(0);
  drv.begin();
  tcaselect(1);
  drv.begin();
    
  drv.selectLibrary(1);
  drv.setMode(DRV2605_MODE_INTTRIG); 


  //Turn off all the sensors so that they can be turned on
  // one at a time to eliminate crosstalk.
  digitalWrite(distPower, LOW);
  digitalWrite(leftPower, LOW);
  digitalWrite(rightPower, LOW);
}

//Main loop where the action takes place
void loop()
{
  int distMode = 0;       //Median for the distance sensor.
  int leftMode = 0;       //Median for the left direction sensor.
  int rightMode = 0;      //Median for the right direction sensor.

  //Get median of distance readings from the left direction sensor.
  digitalWrite(leftPower, HIGH);
  delayMicroseconds(25);
  leftMode = getSensorData(leftSensor);
  digitalWrite(leftPower, LOW);

  //Determine what tactile output to perform with the left direction sensor data.
  if (leftMode < 60)
  {
    solStates[3] = 1;
    //Only send non-repetitve left value through Bluetooth
    if (prevLeft != 1)
    {
      ble.print("L");
    }
    prevLeft = 1;
  }
  else
  {
    solStates[3] = 0;
    prevLeft = 0;
  }

  //Get median of distance readings from the right direction sensor.
  digitalWrite(rightPower, HIGH);
  delayMicroseconds(25);
  rightMode = getSensorData(rightSensor);
  digitalWrite(rightPower, LOW);

  //Determine what tactile output to perform with the right direction sensor data.
  if (rightMode < 60)
  {
    solStates[4] = 1;
    //Only send non-repetitve right value through Bluetooth
    if (prevRight != 1)
    {
      ble.print("R");
    }
    prevRight = 1;
  }
  else
  {
    solStates[4] = 0;
    prevRight = 0;
  }
  actuate();
  
  //Check for Bluetooth input.
  //If there is new Bluetooth input, parse it
  if ( ble.available() )
  {
    char c = ble.read();
  
    //Check turnleft bit
    if(c == 'L')
    {
      solStates[5] = 1;
      solStates[6] = 0;     
    }
    //Check turnright bit
    else if (c == 'R')
    {
      solStates[5] = 0;
      solStates[6] = 1; 
    }
  }
  else
  {
    solStates[5] = 0;
    solStates[6] = 0; 
  }

  //Get median of distance readings from the distance sensor.
  digitalWrite(distPower, HIGH);
  delayMicroseconds(25);
  distMode = getSensorData(distSensor);
  digitalWrite(distPower, LOW);

  //Determine what tactile output to perform with the distance sensor data.
  distFeedback(distMode);
  
  actuate();
}

void distFeedback (int dist)
{
  //Divide the dist by 12 to conver the distance value to feet.
  dist = dist / 12;

  //Determine which 3ft range the detected object is in (0-3ft, 3-6ft, 6-9ft)
  //  and set the appropriate solenodis to be raised/lowered.
  switch (dist / 3)
  {
    case 2:
      solStates[0] = 1;
      solStates[1] = 0; 
      solStates[2] = 0; 
      break;
      
    case 1:
      solStates[0] = 0;
      solStates[1] = 1; 
      solStates[2] = 0;
      break;
      
    case 0:
      solStates[0] = 0;
      solStates[1] = 0; 
      solStates[2] = 1;
      break;
      
    default:
      solStates[0] = 0;
      solStates[1] = 0; 
      solStates[2] = 0;
  }

  //If an object is detected outside of the working range of the sensor,
  //  send a '!' to the connected Bluetooth device.
  if (dist > 8)
  {
    ble.print("!");
  }
  //Otherwise, vibrate the vibration motor with the proper intensity and send
  //  the appropriate distance value to the connected Bluetooth device.
  else
  {
    pulseVibrator(1,dist%3);
    if (dist <= 1)
    {
      ble.print('0');
    }
    //Do not send repeated distance values to the connected Bluetooth device.
    else if (prevDist != dist)
    {
      ble.print(dist);
    }
    prevDist = dist;
  }
}

//Author: Stephen Miles
//Date: 4/17/17
//Function written by a Project EASI group member that allows the solenoids to be
//  properly actuated.
void actuate()
{
  int i = 0;
  for(i=0; i<7; i++)
    digitalWrite(solMap[i], LOW);
  delay(2);
  for(i=0; i<7; i++)
  {
    if(solStates[i] == 1)
      digitalWrite(solMap[i], HIGH);
  }
}

//Function was acquired from an Adafruit guide to using their 
//  Adafruit TCA9548A 1-to-8 I2C Multiplexer Breakout product 
//  which was used in the project.
//Used to control the I2C Multiplexer.
void tcaselect(uint8_t i) 

  if (i > 7) return;
 
  Wire.beginTransmission(TCAADDR);
  Wire.write(1 << i);
  Wire.endTransmission();  
}

//The puleVibrator function pulses the selected vibration motor with the
//  selected intensity.
void pulseVibrator(int vibrator, int intensity)
{
  int effect = 1;
  if(intensity == 0)
  {
    effect = 1;
  }
  else if(intensity == 1)
  {
    effect = 10;
  }
  else if(intensity == 2)
  {
    effect = 12;
  }
  else if(intensity == 3)
  {
    effect = 15;
  }
  
  tcaselect(vibrator);
  drv.setWaveform(0, effect);
  drv.setWaveform(1, 0);
  drv.go();
  
}

//Compare function to use with qsort function.
int compare (const void * a, const void * b)
{
   return ( *(int*)a - *(int*)b );
}

//Function: getSensorData
//Purpose: Gets the mode of pwm sensor input.
//Author: Jason Lessels
//Date created: 2011/June/06
int getSensorData(int pwPin)
{
  int modE;
  pinMode(pwPin, INPUT);

  for (int i = 0; i < arraysize; i++)
  {
    pulse = pulseIn(pwPin, HIGH);
    rangevalue[i] = pulse / 147;
  }

  qsort(rangevalue, arraysize, sizeof(int), compare);
  
  modE = rangevalue[(arraysize / 2)];
  return modE;
}

